/*

oe_v3.h

Simple Vector Class
by orion elenzil

*/


#ifndef	_OE_V3_H_
#define	_OE_V3_H_

#include <math.h>

#ifndef OE_SCALAR
#define	OE_SCALAR
typedef	float	oe_scalar;
#endif

#define	OE_X	0
#define	OE_Y	1
#define	OE_Z	2

class	oe_v3
	{
    public:

    oe_scalar	m_Value[3];

   			   			oe_v3		();
                        oe_v3		(oe_scalar	x, oe_scalar	y, oe_scalar	z);
    void	  			Set	 		(oe_scalar	x, oe_scalar	y, oe_scalar	z);
    void	  			Get	 		(oe_scalar&	x, oe_scalar&	y, oe_scalar&	z);
    oe_scalar*			Get	 		();

	// convenient operators
	oe_v3&	operator	=			(const  oe_v3&		v);

	oe_v3	operator	+			(const	oe_v3&		v);
	oe_v3	operator	-			(const	oe_v3&		v);
	oe_v3	operator	*			(const	oe_v3&		v);
	oe_v3	operator	/			(const	oe_v3&		v);

	oe_v3&	operator	-=			(const	oe_v3&		v);
	oe_v3&	operator	+=			(const	oe_v3&		v);
	oe_v3&	operator	*=			(const	oe_v3&		v);
	oe_v3&	operator	/=			(const	oe_v3&		v);

	oe_v3	operator	*			(		oe_scalar	s);
	oe_v3	operator	/			(		oe_scalar	s);
	oe_v3&	operator	*=			(		oe_scalar	s);
	oe_v3&	operator	/=			(		oe_scalar	s);

	oe_v3				Cross		(const	oe_v3&		v);
	oe_v3&				CrossEq		(const	oe_v3&		v);

	oe_v3				Normalized	();
	oe_v3&				NormalizedEq();

	// inconvenient but faster versions of the same operators.
	void				Plus		(oe_v3&		src,	oe_v3& dst);
	void				Plus2D		(oe_v3&		src,	oe_v3& dst);
	void				Minus		(oe_v3&		src,	oe_v3& dst);
	void				Minus2D		(oe_v3&		src,	oe_v3& dst);
	void				Mult		(oe_v3&		src,	oe_v3& dst);
	void				Mult2D		(oe_v3&		src,	oe_v3& dst);
	void				Div			(oe_v3&		src,	oe_v3& dst);
	void				Div2D		(oe_v3&		src,	oe_v3& dst);
	void				Mult		(oe_scalar	src,	oe_v3& dst);
	void				Mult2D		(oe_scalar	src,	oe_v3& dst);
	void				Div			(oe_scalar	src,	oe_v3& dst);
	void				Div2D		(oe_scalar	src,	oe_v3& dst);
	oe_scalar			Dot			(oe_v3&		src);
	oe_scalar			Dot2D		(oe_v3&		src);
	oe_scalar			Length		();
	oe_scalar			Length2D	();
	void				Norm		(					oe_v3& dst);
	void				Norm2D		(					oe_v3& dst);
	void				Cross		(oe_v3&		src,	oe_v3& dst);

	void				RotX		(oe_scalar	theta,	oe_v3& dst);
	void				RotY		(oe_scalar	theta,	oe_v3& dst);
	void				RotZ		(oe_scalar	theta,	oe_v3& dst);

	// Misc
	void				ToYawPitchRoll	(oe_v3& up, oe_scalar& yaw, oe_scalar& pitch, oe_scalar& roll);



	// Line Routines
	void				ClosestPointOnLine			(oe_v3& line_pt, oe_v3& line_v, oe_v3& dst);
	void				ClosestPointOnLine2D	   	(oe_v3& line_pt, oe_v3& line_v, oe_v3& dst);
	void				ClosestPointOnLine_NoNorm	(oe_v3& line_pt, oe_v3& line_v, oe_v3& dst);
	void				ClosestPointOnLine_NoNorm2D	(oe_v3& line_pt, oe_v3& line_v, oe_v3& dst);
	oe_scalar			DistanceToLine				(oe_v3& line_pt, oe_v3& line_v);
	oe_scalar			DistanceToLine2D			(oe_v3& line_pt, oe_v3& line_v);
	oe_scalar			DistanceToLine_NoNorm		(oe_v3& line_pt, oe_v3& line_v);
	oe_scalar			DistanceToLine_NoNorm2D		(oe_v3& line_pt, oe_v3& line_v);
	void				LineIntersection2D			(oe_v3& Pa, oe_v3& Va, oe_v3& Pb, oe_v3& Vb);

	};

inline	oe_v3::oe_v3()
	{
//	opted not to initialize, as this happens for all the inlines and all.
//	Set(0.0, 0.0, 0.0);
	}

inline	oe_v3::oe_v3(oe_scalar x, oe_scalar y, oe_scalar z)
	{
    Set(x, y, z);
	}


inline	void	oe_v3::Set(oe_scalar x, oe_scalar y, oe_scalar z)
	{
    m_Value[OE_X]	= x;
    m_Value[OE_Y]	= y;
    m_Value[OE_Z]	= z;
    }

inline	void	oe_v3::Get(oe_scalar& x, oe_scalar& y, oe_scalar& z)
	{
    x = m_Value[OE_X];
    y = m_Value[OE_Y];
    z = m_Value[OE_Z];
    }

inline	oe_scalar*	oe_v3::Get()
	{
    return(m_Value);
	}

inline    oe_v3	oe_v3::operator	+	(const oe_v3& v)
	{
    oe_v3	v1;

    v1.m_Value[OE_X] = v.m_Value[OE_X] + m_Value[OE_X];
    v1.m_Value[OE_Y] = v.m_Value[OE_Y] + m_Value[OE_Y];
    v1.m_Value[OE_Z] = v.m_Value[OE_Z] + m_Value[OE_Z];

    return(v1);
    }

inline    oe_v3	oe_v3::operator	-	(const oe_v3& v)
	{
    oe_v3	v1;

    v1.m_Value[OE_X] = m_Value[OE_X] - v.m_Value[OE_X];
    v1.m_Value[OE_Y] = m_Value[OE_Y] - v.m_Value[OE_Y];
    v1.m_Value[OE_Z] = m_Value[OE_Z] - v.m_Value[OE_Z];

    return(v1);
    }

// This is a component-wise multiplication,
// which isn't normally a standard 3D vector operation,
// but i have wanted it from time to time.
// THIS IS NOT DOT PRODUCT !!
inline    oe_v3	oe_v3::operator	*	(const oe_v3& v)
	{
    oe_v3	v1;

    v1.m_Value[OE_X] = v.m_Value[OE_X] * m_Value[OE_X];
    v1.m_Value[OE_Y] = v.m_Value[OE_Y] * m_Value[OE_Y];
    v1.m_Value[OE_Z] = v.m_Value[OE_Z] * m_Value[OE_Z];

    return(v1);
    }

// This is a component-wise division,
// which isn't normally a standard 3D vector operation,
// but i have wanted it from time to time.
inline    oe_v3	oe_v3::operator	/	(const oe_v3& v)
	{
    oe_v3	v1;

    v1.m_Value[OE_X] = m_Value[OE_X] / v.m_Value[OE_X];
    v1.m_Value[OE_Y] = m_Value[OE_Y] / v.m_Value[OE_Y];
    v1.m_Value[OE_Z] = m_Value[OE_Z] / v.m_Value[OE_Z];

    return(v1);
    }

inline    oe_v3&	oe_v3::operator	=	(const oe_v3& v)
	{
    m_Value[OE_X] = v.m_Value[OE_X];
    m_Value[OE_Y] = v.m_Value[OE_Y];
	m_Value[OE_Z] = v.m_Value[OE_Z];

    return(*this);
    }

inline    oe_v3&	oe_v3::operator	+=	(const oe_v3& v)
	{
    m_Value[OE_X] += v.m_Value[OE_X];
    m_Value[OE_Y] += v.m_Value[OE_Y];
    m_Value[OE_Z] += v.m_Value[OE_Z];

    return(*this);
    }

inline    oe_v3&	oe_v3::operator	-=	(const oe_v3& v)
	{
    m_Value[OE_X] -= v.m_Value[OE_X];
    m_Value[OE_Y] -= v.m_Value[OE_Y];
    m_Value[OE_Z] -= v.m_Value[OE_Z];

    return(*this);
    }

// This is a component-wise multiplication,
// which isn't normally a standard 3D vector operation,
// but i have wanted it from time to time.
// THIS IS NOT DOT PRODUCT !!
inline    oe_v3&	oe_v3::operator	*=	(const oe_v3& v)
	{
    m_Value[OE_X] *= v.m_Value[OE_X];
    m_Value[OE_Y] *= v.m_Value[OE_Y];
    m_Value[OE_Z] *= v.m_Value[OE_Z];

    return(*this);
    }

// This is a component-wise division,
// which isn't normally a standard 3D vector operation,
// but i have wanted it from time to time.
inline    oe_v3&	oe_v3::operator	/=	(const oe_v3& v)
	{
    m_Value[OE_X] /= v.m_Value[OE_X];
    m_Value[OE_Y] /= v.m_Value[OE_Y];
    m_Value[OE_Z] /= v.m_Value[OE_Z];

    return(*this);
    }

inline    oe_v3	oe_v3::operator	*	(const oe_scalar s)
	{
    oe_v3	v1;

    v1.m_Value[OE_X] = m_Value[OE_X] * s;
    v1.m_Value[OE_Y] = m_Value[OE_Y] * s;
    v1.m_Value[OE_Z] = m_Value[OE_Z] * s;

	return(v1);
    }

inline    oe_v3	oe_v3::operator	/	(const oe_scalar s)
	{
    oe_v3	v1;

    v1.m_Value[OE_X] = m_Value[OE_X] / s;
    v1.m_Value[OE_Y] = m_Value[OE_Y] / s;
    v1.m_Value[OE_Z] = m_Value[OE_Z] / s;

    return(v1);
    }

inline    oe_v3&	oe_v3::operator	*=	(const oe_scalar s)
	{
    m_Value[OE_X] *= s;
    m_Value[OE_Y] *= s;
    m_Value[OE_Z] *= s;

    return(*this);
    }

inline    oe_v3&	oe_v3::operator	/=	(const oe_scalar s)
	{
    m_Value[OE_X] /= s;
    m_Value[OE_Y] /= s;
	m_Value[OE_Z] /= s;

    return(*this);
    }

// standard right handed cross product
// leaves my data the same.
inline    oe_v3	oe_v3::Cross(const oe_v3& v)
	{
    oe_v3	v1;

    v1.m_Value[OE_X] = m_Value[OE_Y] * v.m_Value[OE_Z] - m_Value[OE_Z] * v.m_Value[OE_Y];
    v1.m_Value[OE_Y] = m_Value[OE_Z] * v.m_Value[OE_X] - m_Value[OE_X] * v.m_Value[OE_Z];
    v1.m_Value[OE_Z] = m_Value[OE_X] * v.m_Value[OE_Y] - m_Value[OE_Y] * v.m_Value[OE_X];

    return(v1);
    }

// standard right handed cross product
// in place. like *= or +=
inline    oe_v3&	oe_v3::CrossEq(const oe_v3& v)
	{
    oe_v3	v1;

    v1 = Cross(v);

    m_Value[OE_X]	= v1.m_Value[OE_X];
    m_Value[OE_Y]	= v1.m_Value[OE_Y];
    m_Value[OE_Z]	= v1.m_Value[OE_Z];

    return(*this);
    }

inline	oe_v3	oe_v3::Normalized()
	{
	oe_v3	v1;
    oe_scalar	length;

    v1 = *this;
    length	= Length();
    if (length > 0.0)
    	v1 /= length;

    return(v1);
    }

inline	oe_v3&	oe_v3::NormalizedEq()
	{
    oe_scalar	length;

    length = Length();

    if (length > 0.0)
    	*this /= length;

    return(*this);
    }


inline	void		oe_v3::Plus		(oe_v3&		src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] + src.m_Value[OE_X];
	dst.m_Value[OE_Y]	= m_Value[OE_Y] + src.m_Value[OE_Y];
	dst.m_Value[OE_Z]	= m_Value[OE_Z] + src.m_Value[OE_Z];
	}

inline	void		oe_v3::Plus2D		(oe_v3&		src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] + src.m_Value[OE_X];
	dst.m_Value[OE_Y]	= m_Value[OE_Y] + src.m_Value[OE_Y];
	}

inline	void		oe_v3::Minus	(oe_v3&		src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] - src.m_Value[OE_X];
	dst.m_Value[OE_Y]	= m_Value[OE_Y] - src.m_Value[OE_Y];
	dst.m_Value[OE_Z]	= m_Value[OE_Z] - src.m_Value[OE_Z];
	}

inline	void		oe_v3::Minus2D	(oe_v3&		src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] - src.m_Value[OE_X];
	dst.m_Value[OE_Y]	= m_Value[OE_Y] - src.m_Value[OE_Y];
	}

inline	void		oe_v3::Mult		(oe_v3&		src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] * src.m_Value[OE_X];
	dst.m_Value[OE_Y]	= m_Value[OE_Y] * src.m_Value[OE_Y];
	dst.m_Value[OE_Z]	= m_Value[OE_Z] * src.m_Value[OE_Z];
	}

inline	void		oe_v3::Mult2D		(oe_v3&		src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] * src.m_Value[OE_X];
	dst.m_Value[OE_Y]	= m_Value[OE_Y] * src.m_Value[OE_Y];
	}

inline	void		oe_v3::Div		(oe_v3&		src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] / src.m_Value[OE_X];
	dst.m_Value[OE_Y]	= m_Value[OE_Y] / src.m_Value[OE_Y];
	dst.m_Value[OE_Z]	= m_Value[OE_Z] / src.m_Value[OE_Z];
	}

inline	void		oe_v3::Div2D		(oe_v3&		src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] / src.m_Value[OE_X];
	dst.m_Value[OE_Y]	= m_Value[OE_Y] / src.m_Value[OE_Y];
	}

inline	void		oe_v3::Mult		(oe_scalar	src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] * src;
	dst.m_Value[OE_Y]	= m_Value[OE_Y] * src;
	dst.m_Value[OE_Z]	= m_Value[OE_Z] * src;
	}

inline	void		oe_v3::Mult2D		(oe_scalar	src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] * src;
	dst.m_Value[OE_Y]	= m_Value[OE_Y] * src;
	}

inline	void		oe_v3::Div		(oe_scalar	src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] / src;
	dst.m_Value[OE_Y]	= m_Value[OE_Y] / src;
	dst.m_Value[OE_Z]	= m_Value[OE_Z] / src;
	}

inline	void		oe_v3::Div2D		(oe_scalar	src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] / src;
	dst.m_Value[OE_Y]	= m_Value[OE_Y] / src;
	}

inline	oe_scalar	oe_v3::Dot		(oe_v3&		src)
	{
	return	(
			m_Value[OE_X] * src.m_Value[OE_X] +
			m_Value[OE_Y] * src.m_Value[OE_Y] +
			m_Value[OE_Z] * src.m_Value[OE_Z]
			);
	}

inline	oe_scalar	oe_v3::Dot2D	(oe_v3&		src)
	{
	return	(
			m_Value[OE_X] * src.m_Value[OE_X] +
			m_Value[OE_Y] * src.m_Value[OE_Y]
			);
	}

inline	oe_scalar	oe_v3::Length	()
	{
	return(sqrt(Dot(*this)));
	}

inline	oe_scalar	oe_v3::Length2D	()
	{
	return(sqrt(Dot2D(*this)));
	}

inline	void		oe_v3::Norm		(					oe_v3& dst)
	{
	oe_scalar	len;

	len	= Length();
	if (len != 0.0)
		Mult(1.0/len, dst);
	}

// Ignores Z component
inline	void		oe_v3::Norm2D		(					oe_v3& dst)
	{
	oe_scalar	len;

	len	= Length2D();
	if (len != 0.0)
		Mult2D(1.0/len, dst);
	}

inline	void		oe_v3::Cross	(oe_v3&		src,	oe_v3& dst)
	{
	dst.m_Value[OE_X] = m_Value[OE_Y] * src.m_Value[OE_Z] - m_Value[OE_Z] * src.m_Value[OE_Y];
	dst.m_Value[OE_Y] = m_Value[OE_Z] * src.m_Value[OE_X] - m_Value[OE_X] * src.m_Value[OE_Z];
	dst.m_Value[OE_Z] = m_Value[OE_X] * src.m_Value[OE_Y] - m_Value[OE_Y] * src.m_Value[OE_X];
	}


inline	void	oe_v3::RotX(oe_scalar theta, oe_v3& dst)
	{
	oe_scalar	s, c;
	
	s	=	sinf(theta);
	c	=	cosf(theta);

	oe_scalar	tmp;

	tmp	=	m_Value[1];

	dst.m_Value[1]	=	tmp			* c	- m_Value[2]	* s;
	dst.m_Value[2]	=	m_Value[2]	* c	+ tmp			* s;
	}


inline	void	oe_v3::RotY(oe_scalar theta, oe_v3& dst)
	{
	oe_scalar	s, c;
	
	s	=	sinf(theta);
	c	=	cosf(theta);

	oe_scalar	tmp;

	tmp	=	m_Value[2];

	dst.m_Value[2]	=	tmp			* c	- m_Value[0]	* s;
	dst.m_Value[0]	=	m_Value[0]	* c	+ tmp			* s;
	}

inline	void	oe_v3::RotZ(oe_scalar theta, oe_v3& dst)
	{
	oe_scalar	s, c;
	
	s	=	sinf(theta);
	c	=	cosf(theta);

	oe_scalar	tmp;

	tmp	=	m_Value[0];

	dst.m_Value[0]	=	tmp			* c	- m_Value[1]	* s;
	dst.m_Value[1]	=	m_Value[1]	* c	+ tmp			* s;
	}

// This takes the unitZ vector with unitY up into this vector with Y as up,
// as represented by y, p, r.
// Roll is rotate about Z, Pitch is about X, Yaw is about Y
// Um,
// This may not quite be working.
inline	void	oe_v3::ToYawPitchRoll	(oe_v3& up, oe_scalar& yaw, oe_scalar& pitch, oe_scalar& roll)
	{
	oe_v3		v1;
	oe_v3		v2;

	v2		=	*this;

	yaw		=	0.0f;
	pitch	=	0.0f;
	roll	=	0.0f;

	////////////////////////
	// Yaw
	//
	// project this onto XZ
	v1.m_Value[0]	=	v2.m_Value[0];
	v1.m_Value[1]	=	0.0f;
	v1.m_Value[2]	=	v2.m_Value[2];
	if (v1.Length() !=	0.0f) {
		v1.Norm(v1);

		// arccos(v1 dot unitZ)
		yaw		=	acosf(v1.m_Value[2]);
		if (v1.m_Value[0] < 0.0)
			yaw	*=	-1.0f;

		// assert yaw
		v2.RotY(-yaw, v2);
		}


	////////////////////////
	// Pitch
	//
	// project this onto YZ
	v1.m_Value[0]	=	0.0f;
	v1.m_Value[1]	=	v2.m_Value[1];
	v1.m_Value[2]	=	v2.m_Value[2];
	if (v1.Length() !=	0.0f) {
		v1.Norm(v1);

		// arccos(v1 dot unitZ)
		pitch	=	acosf(v1.m_Value[2]);
		if (v1.m_Value[1] > 0.0)
			pitch	*=	-1.0f;

		// assert pitch
		v2.RotX(-pitch, v2);
		}

	////////////////////////
	// Roll
	//

	// assert yaw and pitch
	v2	=	up;
	v2.RotY(-yaw	, v2);
	v2.RotX(-pitch	, v2);

	// project this onto XY
	v1.m_Value[0]	=	v2.m_Value[0];
	v1.m_Value[1]	=	v2.m_Value[1];
	v1.m_Value[2]	=	0.0f;
	if (v1.Length() !=	0.0f) {
		v1.Norm(v1);

		// arccos(v1 dot unitY)
		roll	=	acosf(v1.m_Value[1]);
		if (v1.m_Value[0] > 0.0)
			roll	*=	-1.0f;
		}

	}

// line_pt and line_v specify a point and a direction, and hence a line.
// set dst equal to the point on that line closest to THIS.
// Assumes that line_v is normalized.
inline	void	oe_v3::ClosestPointOnLine	(oe_v3& line_pt, oe_v3& line_v, oe_v3& dst)
	{
	oe_v3		v1;
	oe_scalar	s;

	Minus(line_pt, v1);
	s	=	v1.Dot(line_v);
	line_v	.Mult(s, dst);
	line_pt	.Plus(dst, dst);
	}

inline	void	oe_v3::ClosestPointOnLine2D	(oe_v3& line_pt, oe_v3& line_v, oe_v3& dst)
	{
	oe_v3		v1;
	oe_scalar	s;

	Minus2D(line_pt, v1);
	s	=	v1.Dot2D(line_v);
	line_v	.Mult2D(s, dst);
	line_pt	.Plus2D(dst, dst);
	}

// Same as ClosestPointOnLine, excpet line_v need not be normalized.
inline	void	oe_v3::ClosestPointOnLine_NoNorm	(oe_v3& line_pt, oe_v3& line_v, oe_v3& dst)
	{
	oe_v3	v;
	line_v.Norm(v);
	ClosestPointOnLine(line_pt, line_v, dst);
	}

// Same as ClosestPointOnLine, excpet line_v need not be normalized.
inline	void	oe_v3::ClosestPointOnLine_NoNorm2D	(oe_v3& line_pt, oe_v3& line_v, oe_v3& dst)
	{
	oe_v3	v;
	line_v.Norm2D(v);
	ClosestPointOnLine2D(line_pt, line_v, dst);
	}

// Returns shortest distance from THIS to thw given line.
// Assumes line_v is normalized
inline	oe_scalar	oe_v3::DistanceToLine	(oe_v3& line_pt, oe_v3& line_v)
	{
	oe_v3	closestPt;

	ClosestPointOnLine(line_pt, line_v, closestPt);
	Minus(closestPt, closestPt);
	return closestPt.Length();
	}

// Returns shortest distance from THIS to thw given line.
// Assumes line_v is normalized
inline	oe_scalar	oe_v3::DistanceToLine2D	(oe_v3& line_pt, oe_v3& line_v)
	{
	oe_v3	closestPt;

	ClosestPointOnLine2D(line_pt, line_v, closestPt);
	Minus2D(closestPt, closestPt);
	return closestPt.Length();
	}

// Does not Assume line_v is normalized
inline	oe_scalar	oe_v3::DistanceToLine_NoNorm(oe_v3& line_pt, oe_v3& line_v)
	{
	oe_v3	closestPt;

	oe_v3	v;
	line_v.Norm(v);

	ClosestPointOnLine(line_pt, v, closestPt);
	Minus(closestPt, closestPt);
	return closestPt.Length();
	}

// Does not Assume line_v is normalized
inline	oe_scalar	oe_v3::DistanceToLine_NoNorm2D(oe_v3& line_pt, oe_v3& line_v)
	{
	oe_v3	closestPt;

	oe_v3	v;
	line_v.Norm2D(v);

	ClosestPointOnLine2D(line_pt, v, closestPt);
	Minus2D(closestPt, closestPt);
	return closestPt.Length();
	}

// Set THIS to be the intersection of the given lines.
// If the lines are parallel, dst is set to the midpoint between the points.
// Does not need any normalization
// thanks to toby
inline	void	oe_v3::LineIntersection2D(oe_v3& Pa, oe_v3& Va, oe_v3& Pb, oe_v3& Vb)
	{
	oe_scalar	ta;
	oe_scalar	s;

	s	=	Va.m_Value[0] * Vb.m_Value[1] -
			Vb.m_Value[0] * Va.m_Value[1] ;

	if (s == 0.0) {
		Pa.Plus2D(Pb, *this);
		Mult2D(0.5, *this);
		return;
		}

	ta =	( Vb.m_Value[0] * (Pa.m_Value[1] - Pb.m_Value[1]) +
			  Vb.m_Value[1] * (Pb.m_Value[0] - Pa.m_Value[0]) ) / s;

	Va.Mult2D(ta, *this);
	Plus2D(Pa, *this);
	}


#endif	//	_OE_V3_H_

/*** EOF EOF EOF ***/
/*** EOF EOF EOF ***/
/*** EOF EOF EOF ***/
/*** EOF EOF EOF ***/




