/*

oe_v3.h

Simple Vector Class
by orion elenzil

*/


#ifndef	_OE_V3_H_
#define	_OE_V3_H_

#include <math.h>
#include <memory.h>		// for memset, memcpy

#define	OE_DOUBLE	0
#define	OE_SINGLE	1
#define	OE_PRECISION	OE_SINGLE

#if OE_PRECISION	==	OE_DOUBLE
typedef	double	oe_scalar;

#define	oe_sin		sin
#define	oe_cos		cos
#define	oe_tan		tan
#define	oe_asin		asin
#define	oe_acos		acos
#define	oe_atan		atan
#define	oe_pow		pow
#define	oe_sqrt		sqrt
#define oe_abs		abs

#define	OE_SQRT_1D2		0.707106781186547524400844362104849
#define	OE_PID2			1.57079632679489661923132169163975
#define	OE_PI			3.1415926535897932384626433832795
#define	OE_2PI			6.28318530717958647692528676655901
#define	OE_DEG_TO_RAD	OE_2PI / 360
#define	OE_RAD_TO_DEG	360 / OE_2PI
#define	OE_LARGE		1000000000000.0
#define	OE_SMALL		(1.0 / OE_LARGE)

#else
typedef	float	oe_scalar;

#define	oe_sin		sinf
#define	oe_cos		cosf
#define	oe_tan		tanf
#define	oe_asin		asinf
#define	oe_acos		acosf
#define	oe_atan		atanf
#define	oe_pow		powf
#define	oe_sqrt		sqrtf
#define oe_abs		fabsf

#define	OE_SQRT_1D2		0.7071067811865475244008443621048f
#define	OE_PID2			1.5707963267948966192313216916397f
#define	OE_PI			3.1415926535897932384626433832795f
#define	OE_2PI			6.2831853071795864769252867665590f
#define	OE_DEG_TO_RAD	OE_2PI / 360.0f
#define	OE_RAD_TO_DEG	360.0f / OE_2PI
#define	OE_LARGE		1000000000000.0f
#define	OE_SMALL		(1.0f / OE_LARGE)

#endif

#define	OE_MATROW		0	// use matrices as rows of vectors
#define	OE_MATCOL		1	// use matrices as columns of vectors
#define	OE_MATROWCOL	OE_MATROW

#define	OE_X	0
#define	OE_Y	1
#define	OE_Z	2

class	oe_mat3;
class	oe_mat4;

class	oe_v3
	{
    public:

    oe_scalar	m_Value[3];

   			   			oe_v3		();
                        oe_v3		(oe_scalar	x, oe_scalar	y, oe_scalar	z);
    void	  			Set	 		(oe_scalar	x, oe_scalar	y, oe_scalar	z);
    void	  			Set	 		(const	oe_v3&		v);
    void	  			Get	 		(oe_scalar&	x, oe_scalar&	y, oe_scalar&	z);
    oe_scalar*			Get	 		();

	// convenient operators
	oe_v3&	operator	=			(const  oe_v3&		v);

	oe_v3	operator	+			(const	oe_v3&		v);
	oe_v3	operator	-			(const	oe_v3&		v);
	oe_v3	operator	*			(const	oe_v3&		v);
	oe_v3	operator	/			(const	oe_v3&		v);

	oe_v3&	operator	-=			(const	oe_v3&		v);
	oe_v3&	operator	+=			(const	oe_v3&		v);
	oe_v3&	operator	*=			(const	oe_v3&		v);
	oe_v3&	operator	/=			(const	oe_v3&		v);

	oe_v3	operator	*			(		oe_scalar	s);
	oe_v3	operator	/			(		oe_scalar	s);
	oe_v3&	operator	*=			(		oe_scalar	s);
	oe_v3&	operator	/=			(		oe_scalar	s);

	oe_v3				Cross		(const	oe_v3&		v);
	oe_v3&				CrossEq		(const	oe_v3&		v);

	oe_v3				Normalized	();
	oe_v3&				NormalizedEq();

	// inconvenient but faster versions of the same operators.
	void				Plus		(oe_v3&		src,	oe_v3& dst);
	void				Plus2D		(oe_v3&		src,	oe_v3& dst);
	void				Minus		(oe_v3&		src,	oe_v3& dst);
	void				Minus2D		(oe_v3&		src,	oe_v3& dst);
	void				Mult		(oe_v3&		src,	oe_v3& dst);
	void				Mult2D		(oe_v3&		src,	oe_v3& dst);
	void				Div			(oe_v3&		src,	oe_v3& dst);
	void				Div2D		(oe_v3&		src,	oe_v3& dst);
	void				Mult		(oe_scalar	src,	oe_v3& dst);
	void				Mult2D		(oe_scalar	src,	oe_v3& dst);
	void				Div			(oe_scalar	src,	oe_v3& dst);
	void				Div2D		(oe_scalar	src,	oe_v3& dst);
	oe_scalar			Dot			(oe_v3&		src);
	oe_scalar			Dot2D		(oe_v3&		src);
	oe_scalar			Length		();
	oe_scalar			Length2D	();
	void				Norm		(					oe_v3& dst);
	void				Norm2D		(					oe_v3& dst);
	void				Cross		(oe_v3&		src,	oe_v3& dst);

	void				RotX		(oe_scalar	theta,	oe_v3& dst);
	void				RotY		(oe_scalar	theta,	oe_v3& dst);
	void				RotZ		(oe_scalar	theta,	oe_v3& dst);

	// Misc
						// It is highly reccommended to use the RotFromHeading() routine of oe_mat instead of this.
	void				ToYawPitchRoll	(oe_v3& up, oe_scalar& yaw, oe_scalar& pitch, oe_scalar& roll);



	// Line Routines
	void				ClosestPointOnLine			(oe_v3& line_pt, oe_v3& line_v, oe_v3& dst);
	void				ClosestPointOnLine2D	   	(oe_v3& line_pt, oe_v3& line_v, oe_v3& dst);
	void				ClosestPointOnLine_NoNorm	(oe_v3& line_pt, oe_v3& line_v, oe_v3& dst);
	void				ClosestPointOnLine_NoNorm2D	(oe_v3& line_pt, oe_v3& line_v, oe_v3& dst);
	oe_scalar			DistanceToLine				(oe_v3& line_pt, oe_v3& line_v);
	oe_scalar			DistanceToLine2D			(oe_v3& line_pt, oe_v3& line_v);
	oe_scalar			DistanceToLine_NoNorm		(oe_v3& line_pt, oe_v3& line_v);
	oe_scalar			DistanceToLine_NoNorm2D		(oe_v3& line_pt, oe_v3& line_v);
	void				LineIntersection2D			(oe_v3& Pa, oe_v3& Va, oe_v3& Pb, oe_v3& Vb);
	};



class	oe_mat3
	{
	public:

	oe_scalar	m_Value[9];

				oe_mat3				();
	void		Identity			();
	void		RotFromHeading		(oe_v3& heading);
	void		RotFromHeading		(oe_v3& heading, oe_v3& up);
	};


class	oe_mat4
	{
	public:

	oe_scalar	m_Value[16];

				oe_mat4				();
	void		Identity			();
	void		RotFromHeading		(oe_v3& heading);
	void		RotFromHeading		(oe_v3& heading, oe_v3& up);
	};


//
//
// class definitions above, implementations below
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
//
//
//


inline	oe_v3::oe_v3()
	{
//	opted not to initialize, as this happens for all the inlines and all.
//	Set(0.0, 0.0, 0.0);
	}

inline	oe_v3::oe_v3(oe_scalar x, oe_scalar y, oe_scalar z)
	{
    Set(x, y, z);
	}


inline	void	oe_v3::Set(oe_scalar x, oe_scalar y, oe_scalar z)
	{
    m_Value[OE_X]	= x;
    m_Value[OE_Y]	= y;
    m_Value[OE_Z]	= z;
    }

inline	void	oe_v3::Set(const oe_v3& v)
	{
    m_Value[OE_X]	= v.m_Value[OE_X];
    m_Value[OE_Y]	= v.m_Value[OE_Y];
    m_Value[OE_Z]	= v.m_Value[OE_Z];
	}

inline	void	oe_v3::Get(oe_scalar& x, oe_scalar& y, oe_scalar& z)
	{
    x = m_Value[OE_X];
    y = m_Value[OE_Y];
    z = m_Value[OE_Z];
    }

inline	oe_scalar*	oe_v3::Get()
	{
    return(m_Value);
	}

inline    oe_v3	oe_v3::operator	+	(const oe_v3& v)
	{
    oe_v3	v1;

    v1.m_Value[OE_X] = v.m_Value[OE_X] + m_Value[OE_X];
    v1.m_Value[OE_Y] = v.m_Value[OE_Y] + m_Value[OE_Y];
    v1.m_Value[OE_Z] = v.m_Value[OE_Z] + m_Value[OE_Z];

    return(v1);
    }

inline    oe_v3	oe_v3::operator	-	(const oe_v3& v)
	{
    oe_v3	v1;

    v1.m_Value[OE_X] = m_Value[OE_X] - v.m_Value[OE_X];
    v1.m_Value[OE_Y] = m_Value[OE_Y] - v.m_Value[OE_Y];
    v1.m_Value[OE_Z] = m_Value[OE_Z] - v.m_Value[OE_Z];

    return(v1);
    }

// This is a component-wise multiplication,
// which isn't normally a standard 3D vector operation,
// but i have wanted it from time to time.
// THIS IS NOT DOT PRODUCT !!
inline    oe_v3	oe_v3::operator	*	(const oe_v3& v)
	{
    oe_v3	v1;

    v1.m_Value[OE_X] = v.m_Value[OE_X] * m_Value[OE_X];
    v1.m_Value[OE_Y] = v.m_Value[OE_Y] * m_Value[OE_Y];
    v1.m_Value[OE_Z] = v.m_Value[OE_Z] * m_Value[OE_Z];

    return(v1);
    }

// This is a component-wise division,
// which isn't normally a standard 3D vector operation,
// but i have wanted it from time to time.
inline    oe_v3	oe_v3::operator	/	(const oe_v3& v)
	{
    oe_v3	v1;

    v1.m_Value[OE_X] = m_Value[OE_X] / v.m_Value[OE_X];
    v1.m_Value[OE_Y] = m_Value[OE_Y] / v.m_Value[OE_Y];
    v1.m_Value[OE_Z] = m_Value[OE_Z] / v.m_Value[OE_Z];

    return(v1);
    }

inline    oe_v3&	oe_v3::operator	=	(const oe_v3& v)
	{
    m_Value[OE_X] = v.m_Value[OE_X];
    m_Value[OE_Y] = v.m_Value[OE_Y];
	m_Value[OE_Z] = v.m_Value[OE_Z];

    return(*this);
    }

inline    oe_v3&	oe_v3::operator	+=	(const oe_v3& v)
	{
    m_Value[OE_X] += v.m_Value[OE_X];
    m_Value[OE_Y] += v.m_Value[OE_Y];
    m_Value[OE_Z] += v.m_Value[OE_Z];

    return(*this);
    }

inline    oe_v3&	oe_v3::operator	-=	(const oe_v3& v)
	{
    m_Value[OE_X] -= v.m_Value[OE_X];
    m_Value[OE_Y] -= v.m_Value[OE_Y];
    m_Value[OE_Z] -= v.m_Value[OE_Z];

    return(*this);
    }

// This is a component-wise multiplication,
// which isn't normally a standard 3D vector operation,
// but i have wanted it from time to time.
// THIS IS NOT DOT PRODUCT !!
inline    oe_v3&	oe_v3::operator	*=	(const oe_v3& v)
	{
    m_Value[OE_X] *= v.m_Value[OE_X];
    m_Value[OE_Y] *= v.m_Value[OE_Y];
    m_Value[OE_Z] *= v.m_Value[OE_Z];

    return(*this);
    }

// This is a component-wise division,
// which isn't normally a standard 3D vector operation,
// but i have wanted it from time to time.
inline    oe_v3&	oe_v3::operator	/=	(const oe_v3& v)
	{
    m_Value[OE_X] /= v.m_Value[OE_X];
    m_Value[OE_Y] /= v.m_Value[OE_Y];
    m_Value[OE_Z] /= v.m_Value[OE_Z];

    return(*this);
    }

inline    oe_v3	oe_v3::operator	*	(const oe_scalar s)
	{
    oe_v3	v1;

    v1.m_Value[OE_X] = m_Value[OE_X] * s;
    v1.m_Value[OE_Y] = m_Value[OE_Y] * s;
    v1.m_Value[OE_Z] = m_Value[OE_Z] * s;

	return(v1);
    }

inline    oe_v3	oe_v3::operator	/	(const oe_scalar s)
	{
    oe_v3	v1;

    v1.m_Value[OE_X] = m_Value[OE_X] / s;
    v1.m_Value[OE_Y] = m_Value[OE_Y] / s;
    v1.m_Value[OE_Z] = m_Value[OE_Z] / s;

    return(v1);
    }

inline    oe_v3&	oe_v3::operator	*=	(const oe_scalar s)
	{
    m_Value[OE_X] *= s;
    m_Value[OE_Y] *= s;
    m_Value[OE_Z] *= s;

    return(*this);
    }

inline    oe_v3&	oe_v3::operator	/=	(const oe_scalar s)
	{
    m_Value[OE_X] /= s;
    m_Value[OE_Y] /= s;
	m_Value[OE_Z] /= s;

    return(*this);
    }

// standard right handed cross product
// leaves my data the same.
inline    oe_v3	oe_v3::Cross(const oe_v3& v)
	{
    oe_v3	v1;

    v1.m_Value[OE_X] = m_Value[OE_Y] * v.m_Value[OE_Z] - m_Value[OE_Z] * v.m_Value[OE_Y];
    v1.m_Value[OE_Y] = m_Value[OE_Z] * v.m_Value[OE_X] - m_Value[OE_X] * v.m_Value[OE_Z];
    v1.m_Value[OE_Z] = m_Value[OE_X] * v.m_Value[OE_Y] - m_Value[OE_Y] * v.m_Value[OE_X];

    return(v1);
    }

// standard right handed cross product
// in place. like *= or +=
inline    oe_v3&	oe_v3::CrossEq(const oe_v3& v)
	{
    oe_v3	v1;

    v1 = Cross(v);

    m_Value[OE_X]	= v1.m_Value[OE_X];
    m_Value[OE_Y]	= v1.m_Value[OE_Y];
    m_Value[OE_Z]	= v1.m_Value[OE_Z];

    return(*this);
    }

inline	oe_v3	oe_v3::Normalized()
	{
	oe_v3	v1;
    oe_scalar	length;

    v1 = *this;
    length	= Length();
    if (length > 0.0)
    	v1 /= length;

    return(v1);
    }

inline	oe_v3&	oe_v3::NormalizedEq()
	{
    oe_scalar	length;

    length = Length();

    if (length > 0.0)
    	*this /= length;

    return(*this);
    }


inline	void		oe_v3::Plus		(oe_v3&		src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] + src.m_Value[OE_X];
	dst.m_Value[OE_Y]	= m_Value[OE_Y] + src.m_Value[OE_Y];
	dst.m_Value[OE_Z]	= m_Value[OE_Z] + src.m_Value[OE_Z];
	}

inline	void		oe_v3::Plus2D		(oe_v3&		src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] + src.m_Value[OE_X];
	dst.m_Value[OE_Y]	= m_Value[OE_Y] + src.m_Value[OE_Y];
	}

inline	void		oe_v3::Minus	(oe_v3&		src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] - src.m_Value[OE_X];
	dst.m_Value[OE_Y]	= m_Value[OE_Y] - src.m_Value[OE_Y];
	dst.m_Value[OE_Z]	= m_Value[OE_Z] - src.m_Value[OE_Z];
	}

inline	void		oe_v3::Minus2D	(oe_v3&		src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] - src.m_Value[OE_X];
	dst.m_Value[OE_Y]	= m_Value[OE_Y] - src.m_Value[OE_Y];
	}

inline	void		oe_v3::Mult		(oe_v3&		src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] * src.m_Value[OE_X];
	dst.m_Value[OE_Y]	= m_Value[OE_Y] * src.m_Value[OE_Y];
	dst.m_Value[OE_Z]	= m_Value[OE_Z] * src.m_Value[OE_Z];
	}

inline	void		oe_v3::Mult2D		(oe_v3&		src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] * src.m_Value[OE_X];
	dst.m_Value[OE_Y]	= m_Value[OE_Y] * src.m_Value[OE_Y];
	}

inline	void		oe_v3::Div		(oe_v3&		src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] / src.m_Value[OE_X];
	dst.m_Value[OE_Y]	= m_Value[OE_Y] / src.m_Value[OE_Y];
	dst.m_Value[OE_Z]	= m_Value[OE_Z] / src.m_Value[OE_Z];
	}

inline	void		oe_v3::Div2D		(oe_v3&		src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] / src.m_Value[OE_X];
	dst.m_Value[OE_Y]	= m_Value[OE_Y] / src.m_Value[OE_Y];
	}

inline	void		oe_v3::Mult		(oe_scalar	src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] * src;
	dst.m_Value[OE_Y]	= m_Value[OE_Y] * src;
	dst.m_Value[OE_Z]	= m_Value[OE_Z] * src;
	}

inline	void		oe_v3::Mult2D		(oe_scalar	src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] * src;
	dst.m_Value[OE_Y]	= m_Value[OE_Y] * src;
	}

inline	void		oe_v3::Div		(oe_scalar	src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] / src;
	dst.m_Value[OE_Y]	= m_Value[OE_Y] / src;
	dst.m_Value[OE_Z]	= m_Value[OE_Z] / src;
	}

inline	void		oe_v3::Div2D		(oe_scalar	src,	oe_v3& dst)
	{
	dst.m_Value[OE_X]	= m_Value[OE_X] / src;
	dst.m_Value[OE_Y]	= m_Value[OE_Y] / src;
	}

inline	oe_scalar	oe_v3::Dot		(oe_v3&		src)
	{
	return	(
			m_Value[OE_X] * src.m_Value[OE_X] +
			m_Value[OE_Y] * src.m_Value[OE_Y] +
			m_Value[OE_Z] * src.m_Value[OE_Z]
			);
	}

inline	oe_scalar	oe_v3::Dot2D	(oe_v3&		src)
	{
	return	(
			m_Value[OE_X] * src.m_Value[OE_X] +
			m_Value[OE_Y] * src.m_Value[OE_Y]
			);
	}


inline	oe_scalar	oe_v3::Length	()
	{
	return(oe_sqrt(Dot(*this)));
	}

inline	oe_scalar	oe_v3::Length2D	()
	{
	return(oe_sqrt(Dot2D(*this)));
	}

inline	void		oe_v3::Norm		(					oe_v3& dst)
	{
	oe_scalar	len;

	len	= Length();
	if (len != 0.0)
		Mult((oe_scalar)1.0/len, dst);
	}

// Ignores Z component
inline	void		oe_v3::Norm2D		(					oe_v3& dst)
	{
	oe_scalar	len;

	len	= Length2D();
	if (len != 0.0)
		Mult2D(1.0f/len, dst);
	}

inline	void		oe_v3::Cross	(oe_v3&		src,	oe_v3& dst)
	{
	dst.m_Value[OE_X] = m_Value[OE_Y] * src.m_Value[OE_Z] - m_Value[OE_Z] * src.m_Value[OE_Y];
	dst.m_Value[OE_Y] = m_Value[OE_Z] * src.m_Value[OE_X] - m_Value[OE_X] * src.m_Value[OE_Z];
	dst.m_Value[OE_Z] = m_Value[OE_X] * src.m_Value[OE_Y] - m_Value[OE_Y] * src.m_Value[OE_X];
	}


inline	void	oe_v3::RotX(oe_scalar theta, oe_v3& dst)
	{
	oe_scalar	s, c;
	
	s	=	oe_sin(theta);
	c	=	oe_cos(theta);

	oe_scalar	tmp;

	tmp	=	m_Value[1];

	dst.m_Value[1]	=	tmp			* c	- m_Value[2]	* s;
	dst.m_Value[2]	=	m_Value[2]	* c	+ tmp			* s;
	}


inline	void	oe_v3::RotY(oe_scalar theta, oe_v3& dst)
	{
	oe_scalar	s, c;
	
	s	=	oe_sin(theta);
	c	=	oe_cos(theta);

	oe_scalar	tmp;

	tmp	=	m_Value[2];

	dst.m_Value[2]	=	tmp			* c	- m_Value[0]	* s;
	dst.m_Value[0]	=	m_Value[0]	* c	+ tmp			* s;
	}

inline	void	oe_v3::RotZ(oe_scalar theta, oe_v3& dst)
	{
	oe_scalar	s, c;
	
	s	=	oe_sin(theta);
	c	=	oe_cos(theta);

	oe_scalar	tmp;

	tmp	=	m_Value[0];

	dst.m_Value[0]	=	tmp			* c	- m_Value[1]	* s;
	dst.m_Value[1]	=	m_Value[1]	* c	+ tmp			* s;
	}

// This takes the unitZ vector with unitY up into this vector with Y as up,
// as represented by y, p, r.
// Roll is rotate about Z, Pitch is about X, Yaw is about Y
// Um,
// This may not quite be working.
inline	void	oe_v3::ToYawPitchRoll	(oe_v3& up, oe_scalar& yaw, oe_scalar& pitch, oe_scalar& roll)
	{
	oe_v3		v1;
	oe_v3		v2;

	v2		=	*this;

	yaw		=	0.0f;
	pitch	=	0.0f;
	roll	=	0.0f;

	////////////////////////
	// Yaw
	//
	// project this onto XZ
	v1.m_Value[0]	=	v2.m_Value[0];
	v1.m_Value[1]	=	0.0f;
	v1.m_Value[2]	=	v2.m_Value[2];
	if (v1.Length() !=	0.0f) {
		v1.Norm(v1);

		// arccos(v1 dot unitZ)
		yaw		=	oe_acos(v1.m_Value[2]);
		if (v1.m_Value[0] < 0.0)
			yaw	*=	-1.0f;

		// assert yaw
		v2.RotY(-yaw, v2);
		}


	////////////////////////
	// Pitch
	//
	// project this onto YZ
	v1.m_Value[0]	=	0.0f;
	v1.m_Value[1]	=	v2.m_Value[1];
	v1.m_Value[2]	=	v2.m_Value[2];
	if (v1.Length() !=	0.0f) {
		v1.Norm(v1);

		// arccos(v1 dot unitZ)
		pitch	=	oe_acos(v1.m_Value[2]);
		if (v1.m_Value[1] > 0.0)
			pitch	*=	-1.0f;

		// assert pitch
		v2.RotX(-pitch, v2);
		}

	////////////////////////
	// Roll
	//

	// assert yaw and pitch
	v2	=	up;
	v2.RotY(-yaw	, v2);
	v2.RotX(-pitch	, v2);

	// project this onto XY
	v1.m_Value[0]	=	v2.m_Value[0];
	v1.m_Value[1]	=	v2.m_Value[1];
	v1.m_Value[2]	=	0.0f;
	if (v1.Length() !=	0.0f) {
		v1.Norm(v1);

		// arccos(v1 dot unitY)
		roll	=	oe_acos(v1.m_Value[1]);
		if (v1.m_Value[0] > 0.0)
			roll	*=	-1.0f;
		}

	}

// line_pt and line_v specify a point and a direction, and hence a line.
// set dst equal to the point on that line closest to THIS.
// Assumes that line_v is normalized.
inline	void	oe_v3::ClosestPointOnLine	(oe_v3& line_pt, oe_v3& line_v, oe_v3& dst)
	{
	oe_v3		v1;
	oe_scalar	s;

	Minus(line_pt, v1);
	s	=	v1.Dot(line_v);
	line_v	.Mult(s, dst);
	line_pt	.Plus(dst, dst);
	}

inline	void	oe_v3::ClosestPointOnLine2D	(oe_v3& line_pt, oe_v3& line_v, oe_v3& dst)
	{
	oe_v3		v1;
	oe_scalar	s;

	Minus2D(line_pt, v1);
	s	=	v1.Dot2D(line_v);
	line_v	.Mult2D(s, dst);
	line_pt	.Plus2D(dst, dst);
	}

// Same as ClosestPointOnLine, excpet line_v need not be normalized.
inline	void	oe_v3::ClosestPointOnLine_NoNorm	(oe_v3& line_pt, oe_v3& line_v, oe_v3& dst)
	{
	oe_v3	v;
	line_v.Norm(v);
	ClosestPointOnLine(line_pt, line_v, dst);
	}

// Same as ClosestPointOnLine, excpet line_v need not be normalized.
inline	void	oe_v3::ClosestPointOnLine_NoNorm2D	(oe_v3& line_pt, oe_v3& line_v, oe_v3& dst)
	{
	oe_v3	v;
	line_v.Norm2D(v);
	ClosestPointOnLine2D(line_pt, line_v, dst);
	}

// Returns shortest distance from THIS to thw given line.
// Assumes line_v is normalized
inline	oe_scalar	oe_v3::DistanceToLine	(oe_v3& line_pt, oe_v3& line_v)
	{
	oe_v3	closestPt;

	ClosestPointOnLine(line_pt, line_v, closestPt);
	Minus(closestPt, closestPt);
	return closestPt.Length();
	}

// Returns shortest distance from THIS to thw given line.
// Assumes line_v is normalized
inline	oe_scalar	oe_v3::DistanceToLine2D	(oe_v3& line_pt, oe_v3& line_v)
	{
	oe_v3	closestPt;

	ClosestPointOnLine2D(line_pt, line_v, closestPt);
	Minus2D(closestPt, closestPt);
	return closestPt.Length();
	}

// Does not Assume line_v is normalized
inline	oe_scalar	oe_v3::DistanceToLine_NoNorm(oe_v3& line_pt, oe_v3& line_v)
	{
	oe_v3	closestPt;

	oe_v3	v;
	line_v.Norm(v);

	ClosestPointOnLine(line_pt, v, closestPt);
	Minus(closestPt, closestPt);
	return closestPt.Length();
	}

// Does not Assume line_v is normalized
inline	oe_scalar	oe_v3::DistanceToLine_NoNorm2D(oe_v3& line_pt, oe_v3& line_v)
	{
	oe_v3	closestPt;

	oe_v3	v;
	line_v.Norm2D(v);

	ClosestPointOnLine2D(line_pt, v, closestPt);
	Minus2D(closestPt, closestPt);
	return closestPt.Length();
	}

// Set THIS to be the intersection of the given lines.
// If the lines are parallel, dst is set to the midpoint between the points.
// Does not need any normalization
// thanks to toby
inline	void	oe_v3::LineIntersection2D(oe_v3& Pa, oe_v3& Va, oe_v3& Pb, oe_v3& Vb)
	{
	oe_scalar	ta;
	oe_scalar	s;

	s	=	Va.m_Value[0] * Vb.m_Value[1] -
			Vb.m_Value[0] * Va.m_Value[1] ;

	if (s == 0.0) {
		Pa.Plus2D(Pb, *this);
		Mult2D(0.5, *this);
		return;
		}

	ta =	( Vb.m_Value[0] * (Pa.m_Value[1] - Pb.m_Value[1]) +
			  Vb.m_Value[1] * (Pb.m_Value[0] - Pa.m_Value[0]) ) / s;

	Va.Mult2D(ta, *this);
	Plus2D(Pa, *this);
	}

//
//
// vector class above
///////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////
// matrix classes below
//
//



inline	oe_mat3::oe_mat3()
	{
	Identity();
	}

inline	oe_mat4::oe_mat4()
	{
	Identity();
	}

inline	void oe_mat3::Identity()
	{
	memset((void*)(m_Value + 1), 0, sizeof(oe_scalar) * 7);
	m_Value[0]	=	1.0f;
	m_Value[4]	=	1.0f;
	m_Value[8]	=	1.0f;
	}

inline	void oe_mat4::Identity()
	{
	memset((void*)(m_Value + 1), 0, sizeof(oe_scalar) * 14);
	m_Value[ 0]	=	1.0f;
	m_Value[ 5]	=	1.0f;
	m_Value[10]	=	1.0f;
	m_Value[15]	=	1.0f;
	}


inline	void	oe_mat3::RotFromHeading(oe_v3& heading)
	{
	oe_v3	up;

	up.Set(0.0f, 1.0f, 0.0f);
	RotFromHeading(heading, up);
	}

// Create a rotation matrix which will rotate unitZ into heading, unitY into up, and unitX into their cross.
inline	void	oe_mat3::RotFromHeading(oe_v3& vz, oe_v3& vy)
	{
	oe_v3	vx;

	vy.Cross(vz, vx);

#if OE_MATROWCOL == OE_MATROW
	memcpy((void*)(m_Value    ), (void*)(vx.m_Value), sizeof(oe_scalar) * 3);
	memcpy((void*)(m_Value + 3), (void*)(vy.m_Value), sizeof(oe_scalar) * 3);
	memcpy((void*)(m_Value + 6), (void*)(vz.m_Value), sizeof(oe_scalar) * 3);
#else
	oe_scalar* ps;
	ps		=	m_Value;
	*(ps++)	=	vx.m_Value[0];	
	*(ps++)	=	vy.m_Value[0];	
	*(ps++)	=	vz.m_Value[0];	
	*(ps++)	=	vx.m_Value[1];	
	*(ps++)	=	vy.m_Value[1];	
	*(ps++)	=	vz.m_Value[1];	
	*(ps++)	=	vx.m_Value[2];	
	*(ps++)	=	vy.m_Value[2];	
	*(ps++)	=	vz.m_Value[2];	
#endif

	}

inline	void	oe_mat4::RotFromHeading(oe_v3& heading)
	{
	oe_v3	up;

	up.Set(0.0f, 1.0f, 0.0f);
	RotFromHeading(heading, up);
	}

// Create a rotation matrix which will rotate unitZ into heading, unitY into up, and unitX into their cross.
inline	void	oe_mat4::RotFromHeading(oe_v3& vz, oe_v3& vy)
	{
	oe_v3	vx;

	vy.Cross(vz, vx);

#if OE_MATROWCOL == OE_MATROW
	memcpy((void*)(m_Value     ), (void*)(vx.m_Value)	, sizeof(oe_scalar) * 3);
	memcpy((void*)(m_Value +  4), (void*)(vy.m_Value)	, sizeof(oe_scalar) * 3);
	memcpy((void*)(m_Value +  8), (void*)(vz.m_Value)	, sizeof(oe_scalar) * 3);
	memset((void*)(m_Value + 12), 0						, sizeof(oe_scalar) * 3);
	m_Value[ 3]	=	0.0;
	m_Value[ 7]	=	0.0;
	m_Value[11]	=	0.0;
	m_Value[15]	=	1.0;
#else
	oe_scalar* ps;
	ps		=	m_Value;
	*(ps++)	=	vx.m_Value[0];	
	*(ps++)	=	vy.m_Value[0];	
	*(ps++)	=	vz.m_Value[0];	
	*(ps++)	=	0.0;	
	*(ps++)	=	vx.m_Value[1];	
	*(ps++)	=	vy.m_Value[1];	
	*(ps++)	=	vz.m_Value[1];	
	*(ps++)	=	0.0;	
	*(ps++)	=	vx.m_Value[2];	
	*(ps++)	=	vy.m_Value[2];	
	*(ps++)	=	vz.m_Value[2];	
	*(ps++)	=	0.0;	
	memset((void*)(m_Value + 12), 0						, sizeof(oe_scalar) * 3);
	m_Value[15]	=	1.0;
#endif

	}


// returns number of real roots. (0 or 2 or possibly 1)
static	int	oe_Quadratic(oe_scalar A, oe_scalar B, oe_scalar C, oe_scalar& r1, oe_scalar& r2)
	{
	oe_scalar	bsquaredminus4ac;
	oe_scalar	twoa;
	oe_scalar	sqrtstuff;


	if (A == 0.0f)
		return 0;

	twoa	=	2.0f * A;

	bsquaredminus4ac = B * B - 4.0f * A * C;

	if (bsquaredminus4ac == 0.0f)	// unlikely
		{
		r1	=	-B / twoa;
		r2	=	r1;
		return	1;
		}
	else if (bsquaredminus4ac < 0.0f)
		{
		return	0;
		}


	sqrtstuff	=	oe_sqrt(bsquaredminus4ac);
	r1			=	(-B + sqrtstuff) / twoa;
	r2			=	(-B - sqrtstuff) / twoa;

	return 2;
	}



#endif	//	_OE_V3_H_

/*** EOF EOF EOF ***/
/*** EOF EOF EOF ***/
/*** EOF EOF EOF ***/
/*** EOF EOF EOF ***/




