/*

  oe_Timer.h

  Simple Timer Class.

  This is intended to be a fairly global object,
  perhaps one in the HapScene, and a pointer
  to that one given to many objects.
  
  It keeps track of the current time in seconds.

  Usage Notes
  -----------
  To Use this Object to get the current time:
  (By default, the system seconds since the object was instantiated)
  oe_timer	x;
  oe_scalar	t;
  ...
  x.Pulse();
  t = x.GetSeconds();
  ...

  To force this timer to achieve 1 second every 10 times Pulse is called:
  (Useful for non realtime things like high quality rendering)
  x.SetMode(HT_STEPTIME);
  ...
  x.Pulse();
  t = x.GetSeconds();


  SO
  Pulse() should be called ONCE PER FRAME!
  CurrentSeconds() can be called as often as you please,
  and it will return the time of the previous Pulse().


  Future Coding Notes
  -------------------
  Perhaps a callback option for SystemSecs() would be appropriate
  in some cases.

  When porting to other operating systems / methods,
  it is intended that only the SystemSeconds() routine
  need be rewritten.
  
  Currently has no .cpp file.

  */

#ifndef HAPTIMER_H
#define HAPTIMER_H

#define		HT_REALTIME	0
#define		HT_STEPTIME	1
#define		HT_FIXEDTIME 2

#include "osheader.h"
#include "oe_v3.h"
#include "time.h"



class	oe_timer
	{
	protected:
		double	m_SystemSecondsStart;	// Start of time.
		double	m_SystemSecondsCurrent;	// this - start = current.
		double	m_SecondsCurrent;		// The global "Current Seconds"
		double	m_SecondsStep;			// For timestep mode.
		int		m_Mode;					// Realtime or Stepped time or ..
		double  m_MostRecentTimeStep;

#if	OS_BUILD == OS_IRIX


		struct timeval m_TimeInfo;
		struct timezone m_TimeZone;	// not really used

#endif	
		
	public:
		oe_timer();

		oe_scalar	GetSeconds(){return((oe_scalar)m_SecondsCurrent);}
		oe_scalar	GetTimeStep() { return (oe_scalar) m_MostRecentTimeStep; }

		void		SetSeconds(oe_scalar seconds);

		void		SetStep(oe_scalar secondsStep){m_SecondsStep = (double)secondsStep;}

		oe_scalar	GetStep(){return((oe_scalar)m_SecondsStep);}

		double		SystemSeconds();
		void		Pulse();
		void		Increment(oe_scalar delta)
			{
		
			m_SystemSecondsCurrent	+=	delta;
			m_MostRecentTimeStep = delta;

			m_SecondsCurrent = m_SystemSecondsCurrent - m_SystemSecondsStart;
			}
		void		SetMode(int mode){m_Mode	= mode;}

		int		GetMode()	{	return(m_Mode);	}


	protected:


	};

#endif
